/**
 * \file: exchnd_server.c
 *
 * Server code for Health management notification.
 *
 * This code is marked as deprecated and can be removed at any time.
 * It creates a socket and accepts clients on it. If enabled, each registered
 * client will receive a notification when an event occurs. Any data send to
 * this server by the client is dropped.
 *
 * \component: exchndd
 *
 * \author: Frederic Berat (fberat@de.adit-jv.com)
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/
/* PRQA: Lint Message 754: Library macro. */
/*lint -e754 */

#include <errno.h>
#include <pthread.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ptrace.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/un.h>
#include <sys/wait.h>
#include <unistd.h>

#include <linux/exchnd.h>

#include "exchndd.h"
#include "exchnd_backend.h"

#define BIN_PATH "/usr/bin"

#define EXEC_PATH_STR   "--execpath="
#define PID_STR         "--pid=xxxxxxxxxx"
#define PID_TMPL        "--pid=%d"
#define PIDNS_STR       "--pidns=0xxxxxxxxxxxxxxxxx"
#define PIDNS_TMPL      "--pidns=0x%lx"
#define PIDNS_PATH      "/proc/%d/ns/pid"
#define RESULT_STR      "SERVICE_RESULT=signal"
#define CODE_STR        "EXIT_CODE=killed"
#define STATUS_STR      "EXIT_STATUS=xxxxxxxxxx"
#define STATUS_TMPL     "EXIT_STATUS=%s"

#define ARRAY_SIZE(arr) (sizeof(arr) / sizeof((arr)[0]))

#define APP_NAME_MAX_SIZE   4096
struct exchnd_msg {
    pid_t pid;
    int signal;
    ino_t pid_ns;
    char app_name[APP_NAME_MAX_SIZE];
};

static char client_path[APP_NAME_MAX_SIZE];
static char *client_name;

static pthread_t exchnd_server_th;
static pthread_mutex_t mq_lck = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t new_data = PTHREAD_COND_INITIALIZER;

static struct msg_list *mqlist;
static struct msg_list mqlist_head;

const char *exchnd_sig_names[] = EXCHND_SIGNAL_STRINGS;
/*
 * \func signal_name
 *
 * Converts signal number to module name.
 * The function returns the name of the given signal number. If an invalid
 * signal number  is given the function returns the none signal name
 *
 * \param module Module enum value to convert
 *
 * \return String describing the module
 */
static const char *signal_name(int sig)
{
    if ((unsigned int)sig < ARRAY_SIZE(exchnd_sig_names))
        /* + 3 to ignore "SIG" */
        return exchnd_sig_names[sig] + 3;
    else
        /* Return empty signal NONE */
        return exchnd_sig_names[0];
}

void set_client_name(char *in)
{
    char *p = NULL;

    if (!in)
        goto error;

    if (strchr(in, '.')) {
        exchnd_print_error("No '.' allowed in client application.\n");
        goto error;
    }

    if (*in == '/')
        in++;

    p = strrchr(in, '/');

    if (p)
        in = p;

    snprintf(client_path, APP_NAME_MAX_SIZE, BIN_PATH "/%s", in);

    client_name = strdup(in);

    if (access(client_path, X_OK) < 0) {
        exchnd_print_error("\"%s\" do not exist or is not executable (%s).\n",
                           client_path,
                           strerror(errno));
        goto error;
    }

    return;

error:
    memset(client_path, 0, APP_NAME_MAX_SIZE);
    free(client_name);
    client_name = NULL;
    action &= ~EXCHND_SERVER;
    return;
}

/* PRQA: Lint Message 826: Nothing suspicious here. */
/*lint -save -e826 */
static void run_child(struct msg_list *cur)
{
    struct exchnd_msg *msg = (struct exchnd_msg *)cur->msg;

    char *argv[4] = { NULL };
    char *envp[4] = { NULL };
    int len = strlen(msg->app_name) + 1 + sizeof(EXEC_PATH_STR);
    argv[1] = calloc(1, len);
    argv[2] = calloc(1, sizeof(PID_STR));
    argv[3] = calloc(1, sizeof(PIDNS_STR));

    if (!argv[1] || !argv[2] || !argv[3]) {
        free(argv[1]);
        free(argv[2]);
        free(argv[3]);
        return;
    }

    argv[0] = strdup(client_name);
    snprintf(argv[1],
             len,
             EXEC_PATH_STR "%s",
             ((struct exchnd_msg *)cur->msg)->app_name);
    snprintf(argv[2],
             sizeof(PID_STR),
             PID_TMPL,
             ((struct exchnd_msg *)cur->msg)->pid);
    snprintf(argv[3],
             sizeof(PIDNS_STR),
             PIDNS_TMPL,
             ((struct exchnd_msg *)cur->msg)->pid_ns);

    envp[0] = strdup(RESULT_STR);
    envp[1] = strdup(CODE_STR);
    envp[2] = calloc(1, sizeof(STATUS_STR));

    if (!argv[0] || !argv[1] || !argv[2] || !argv[2] ||
        !envp[0] || !envp[1] || !envp[2]) {
        free(argv[0]);
        free(argv[1]);
        free(argv[2]);
        free(argv[3]);
        free(envp[0]);
        free(envp[1]);
        free(envp[2]);
        return;
    }

    snprintf(envp[2],
             sizeof(STATUS_STR),
             STATUS_TMPL,
             signal_name(((struct exchnd_msg *)cur->msg)->signal));

    execve(client_path, argv, envp);

    exchnd_print_error("Error while executing server: %s\n", strerror(errno));
    abort();
}
/*lint -restore */

static void exchnd_send_mesg(void)
{
    struct msg_list *next = mqlist->next;

    while (next != mqlist) {
        struct msg_list *cur = next;
        pid_t pid = 0;
        next = next->next;

        exchnd_list_del(cur);
        pid = fork();

        if (!pid) {
            run_child(cur);
        } else {
            int status = 0;

            do
                waitpid(pid, &status, WUNTRACED | WCONTINUED);
            while (!WIFEXITED(status) && !WIFSIGNALED(status));
        }

        free(cur->msg);
        free(cur);
    }
}

static int get_pidns(pid_t pid)
{
    char buf[sizeof(PIDNS_PATH) + 10] = { 0 };
    struct stat st = { 0 };

    snprintf(buf, sizeof(PIDNS_PATH) + 10, PIDNS_PATH, pid);

    if (stat(buf, &st) < 0)
        return 0;

    return st.st_ino;
}

#define MAX_RETRIES 100
static int get_signal(pid_t pid)
{
    siginfo_t siginfo = { 0 };
    struct ptrace_peeksiginfo_args arg = {
        .nr = 1
    };
    int ret = 0;
    int status = 0;
    int trial = MAX_RETRIES;

    waitpid(pid, &status, WCONTINUED | WUNTRACED);

    ret = ptrace(PTRACE_PEEKSIGINFO, pid, &arg, &siginfo);

    while ((ret < 0) && trial--) {
        if ((trial == MAX_RETRIES / 2) || (trial == 1))
            exchnd_print_error("Failed to peek siginfo (%s).",
                               strerror(errno));

        usleep(10000);
        ret = ptrace(PTRACE_PEEKSIGINFO, pid, &arg, &siginfo);
    }

    if ((siginfo.si_signo == 0) && !ret) {
        trial = MAX_RETRIES;
        arg.flags = PTRACE_PEEKSIGINFO_SHARED;
        ret = ptrace(PTRACE_PEEKSIGINFO, pid, &arg, &siginfo);

        while ((ret < 0) && trial--) {
            if ((trial == MAX_RETRIES / 2) || (trial == 1))
                exchnd_print_error("Failed to peek siginfo (%s).",
                                   strerror(errno));

            usleep(10000);
            ret = ptrace(PTRACE_PEEKSIGINFO, pid, &arg, &siginfo);
        }
    }

    return siginfo.si_signo;
}

void exchnd_server_add_msg(pid_t pid, char *name)
{
    struct exchnd_msg *msg = NULL;
    struct msg_list *next_mlist = NULL;

    if (!name) {
        exchnd_log_error("No application name to send.");
        return;
    }

    msg = calloc(1, sizeof(*msg));

    if (!msg)
        return;

    next_mlist = calloc(1, sizeof(*next_mlist));

    if (!next_mlist) {
        free(msg);
        return;
    }

    next_mlist->msg = (char *)msg;
    msg->signal = get_signal(pid);
    msg->pid_ns = get_pidns(pid);
    msg->pid = pid;
    next_mlist->len = sizeof(*msg);
    snprintf(msg->app_name, APP_NAME_MAX_SIZE, "%s", name);

    pthread_mutex_lock(&mq_lck);
    exchnd_list_add_tail(next_mlist, mqlist);
    pthread_cond_signal(&new_data);
    pthread_mutex_unlock(&mq_lck);
}

static void *exchnd_server(void *arg)
{
    (void)arg;

    mqlist = &mqlist_head;
    exchnd_init_list(mqlist);

    while (!(action & EXCHND_TERMINATE)) {
        pthread_mutex_lock(&mq_lck);

        exchnd_log_error("Handling message list.\n");
        exchnd_send_mesg();

        pthread_cond_wait(&new_data, &mq_lck);
        pthread_mutex_unlock(&mq_lck);
    }

    return NULL;
}

int exchnd_init_server(void)
{
    int ret = 0;

    ret = exchnd_create_thread(&exchnd_server_th,
                               exchnd_server,
                               NULL,
                               EXH_NORMAL_TH);

    if (ret)
        exchnd_print_error("Unable to initialize server: %s", strerror(errno));

    return ret;
}

void exchnd_cleanup_server(void)
{
    pthread_mutex_lock(&mq_lck);
    pthread_cond_signal(&new_data);
    pthread_mutex_unlock(&mq_lck);

    pthread_join(exchnd_server_th, NULL);

    free(client_name);
    client_name = NULL;
}